<?php
/*--------------------------------------------------------------
   GetAvailableWidgets.php 2021-07-12
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Action;

use Exception;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;
use Gambio\Admin\Modules\Statistics\Services\StatisticsOverviewService;
use Gambio\Core\Application\Http\AbstractAction;
use Gambio\Core\Application\Http\Request;
use Gambio\Core\Application\Http\Response;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Language\App\LanguageService;

/**
 * @codeCoverageIgnore
 */
class GetAvailableWidgets extends AbstractAction
{
    private const FALLBACK_LANGUAGE_CODE = "de";
    
    /**
     * @var StatisticsOverviewService
     */
    private $service;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var LanguageService
     */
    private $languageService;
    
    /**
     * @var string
     */
    private $languageCode;
    
    
    /**
     * Constructor.
     */
    public function __construct(
        StatisticsOverviewService $service,
        LanguageService $languageService,
        UserPreferences $userPreferences
    ) {
        $this->service         = $service;
        $this->languageService = $languageService;
        $this->userPreferences = $userPreferences;
        
        try {
            $this->languageCode = $this->languageService->getLanguageById($this->userPreferences->languageId())->code();
        } catch (Exception $exception) {
            $this->languageCode = self::FALLBACK_LANGUAGE_CODE;
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function handle(Request $request, Response $response): Response
    {
        try {
            return $response->withJson(array_map(function (WidgetDefinition $widgetDefinition) {
                return [
                    'type'    => $widgetDefinition->type()->value(),
                    'icon'    => $widgetDefinition->icon()->value(),
                    'title'   => $widgetDefinition->titles()->getByLanguageCode($this->languageCode)->value(),
                    'version' => $widgetDefinition->version()->value(),
                    'author'  => $widgetDefinition->author()->value(),
                ];
            },
                iterator_to_array($this->service->getWidgetDefinitions())));
        } catch (Exception $exception) {
            return $response->withStatus(400)->withJson($exception->getMessage());
        }
    }
}